{
Copyright (c) 2013 Yvon Massé
Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
}

unit Unit3;
{$MODE delphi}

interface

uses
  SysUtils, Classes, Graphics, Controls, Forms, StdCtrls, Math;

type  { TForm3 }
  TForm3 = class(TForm)
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    CmbSo: TComboBox;
    ChkHi: TCheckBox;
    ChkCo: TCheckBox;
    ChkHe: TCheckBox;
    EdHi: TEdit;
    EdCo: TEdit;
    EdHe: TEdit;
    UniHi: TComboBox;
    UniCo: TComboBox;
    ChkHoTh: TCheckBox;
    ChkDbl: TCheckBox;
    ChkPrlx: TCheckBox;
    BtPar: TButton;
    BtCalcul: TButton;
    BtRaz: TButton;
    procedure ChkHiClick(Sender: TObject);
    procedure ChkCoClick(Sender: TObject);
    procedure ChkHeClick(Sender: TObject);
    procedure EdHiExit(Sender: TObject);
    procedure EdCoExit(Sender: TObject);
    procedure EdHeExit(Sender: TObject);
    procedure UniHiChange(Sender: TObject);
    procedure UniCoChange(Sender: TObject);
    procedure ChkHoThClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure BtParClick(Sender: TObject);
    procedure BtRazClick(Sender: TObject);
    procedure BtCalculClick(Sender: TObject);
    function FormHelp({%H-}Command: Word; {%H-}Data: PtrInt; var {%H-}CallHelp: Boolean
      ): Boolean;
    procedure FormCreate(Sender: TObject);
  private
    { private declarations }
  public
    { public declarations }
  end; 

const
  MHaut_0 = 'Valeur : nombre d’itération maxi';
  MHaut_1 = 'Hauteur vraie non observable';
  MHaut_2 = 'Hauteur observée sous l’horizon';
  MHaut_3 = 'Résultat hors de ses limites';
  MHaut_4 = 'Hauteur vraie supérieure à 90°';
  MHaut_5 = 'Paramètres réfraction';
  MHaut_6 = 'Température [°C]';
  MHaut_7 = 'Pression [HPa]';
  MHaut_8 = 'Longueur d’onde [nm]';
  MHaut_9 = 'Humidité relative [%]';
  MHaut_10 = 'Température inchangée'#13#10'Valeur attendue entre -20 et 40 °C';
  MHaut_11 = 'Pression inchangée'#13#10'Valeur attendue entre 500 et 1050 HPa';
  MHaut_12 = 'Longueur d’onde inchangée'#13#10'Valeur attendue entre 400 et 800 nm';
  MHaut_13 = 'Humidité relative inchangée'#13#10'Valeur attendue entre 0 et 100 %';
{
  MHaut_14 = ;
  MHaut_15 = ;
  MHaut_16 = ;
  MHaut_17 = ;
  MHaut_18 = ;
  MHaut_19 = ;
  MHaut_20 = ;
}
  mMHaut = 13;

  RTT = 6378120;         // Rayon terrestre en m
  ZKV = 273.15;          // Zero absolu en °C
  PAN = 1013.25;         // Pression atmosphérique normale en mb
  altm = 3000;           // Altitude maxi d'observation

var
  Form3: TForm3;
  hr: Integer = 20;
  temp: Real = 15;
  pres: Real = PAN;
  lamb: Real = 590;
  humi: Real = 40;
  Atm: Array[0..12] of Real;
  MHaut: Array[0..mMHaut] of String =
    (MHaut_0, MHaut_1, MHaut_2, MHaut_3, MHaut_4, MHaut_5, MHaut_6, MHaut_7, MHaut_8, MHaut_9,
    MHaut_10, MHaut_11, MHaut_12, MHaut_13//, MHaut_14, MHaut_15, MHaut_16, MHaut_17, MHaut_18, MHaut_19,
    );

implementation

uses Unit1;

{$R *.lfm} { TForm3 }

procedure majAtm(H0: Real);
const
  PH = 45;             // Latitude
  AL = 0.0065;         // en °K/m, valeur positive
  GCR = 8314.36;       // Constante des gaz en mJ.mol-1.K-1
  MD = 28.966;         // Masse molaire air sec en g
  MW = 18.016;         // Masse molaire vapeur d'eau en g
  GAMMA = 18.36;
  Z2 = 11.2684e-06;
  HT = 11000;          // Hauteur de la Troposhère en m
var
  GB, Z1, PW0, PP: Real;
begin
  // Set up parameters defined at the observer for the atmosphere
  GB := 9.784*(1 - 0.0026*Cos(2*DegToRad(PH)) - 0.00000028*H0);
  Z1 := (287.604e-6 + 1.6288/(lamb*lamb) + 0.0136e6/(lamb*lamb*lamb*lamb))*ZKV/PAN;
//  Atm[0] := TT  Température de la tropopause, est déterminé dans IntRef()
  Atm[1] := AL;
  Atm[2] := GB*MD/GCR;
  Atm[3] := Atm[2]/Atm[1];
  Atm[4] := GAMMA;
  Atm[5] := RTT + H0;
  Atm[6] := temp + ZKV;
  PW0 := humi*Power(Atm[6]/247.1, Atm[4])/100;
  PP := PW0*(1 - MW/MD)*Atm[3]/(Atm[4] - Atm[3]);
  Atm[7] := Z1*(pres + PP)/Atm[6];
  Atm[8] := (Z1*PP + Z2*PW0)/Atm[6];
  Atm[9] := (Atm[3] - 1)*Atm[1]*Atm[7]/Atm[6];
  Atm[10] := (Atm[4] - 1)*Atm[1]*Atm[8]/Atm[6];
  Atm[11] := RTT + HT;
  Atm[12] := Atm[6] - Atm[1]*(HT - H0);
end;

// Input:
// K  1 : Troposphère; autre valeur : Statosphère
// R  The current distance from the centre of the Earth in meters
// Output:
// N  The refractive index at R
// DNDR The rate the refrative index is changing at R
// Return:
// The refraction integrand

function ATMO(K: Integer; R: Real; var N, DNDR: Real): Real;
var
  TT0, TT01, TT02: Real;
begin
  if K = 1 then
  begin
    TT0 := (Atm[6] - Atm[1]*(R - Atm[5]))/Atm[6];
    TT01 := Power(TT0, Atm[3] - 2);
    TT02 := Power(TT0, Atm[4] - 2);
    N := 1 + (Atm[7]*TT01 - Atm[8]*TT02)*TT0;
    DNDR := -Atm[9]*TT01 + Atm[10]*TT02;
  end
  else
  begin
    TT0 := Atm[2]/Atm[12];
    N := 1 + (Atm[0] - 1)*exp(-TT0*(R - Atm[11]));
    DNDR := -TT0*(Atm[0] - 1)*exp(-TT0*(R - Atm[11]));
  end;
  Result := R*DNDR/(N + R*DNDR);
end;

// A program to calculate bending refraction. The method is based on
// N.A.O Technical Notes 59 and 63 and a paper by Standish and Auer
// ’Astronomical Refraction: Computational Method for all Zenith Angles’.
// Input:
// Z0 The observed zenith distance of the object in degrees
// Output:
// REF The refraction at the observer in degrees

function IntRef(Z0, H0: Real): Real;
const
  RS = RTT + 80000;      // Rayon pour lequel la réfraction est négligeable en m
  DGR = 0.01745329252;   // Pour conversion degrés <--> radian
  EPS = 0.01;
var
  N: Real = 0;
  DNDR: Real = 0;
  SK0, ZT: Real;
  Z, R: Real;
  FB, FE, FO, FF, H: Real;
  REFO, REFP, REFT: Real;
  K, I, J, IT, ISTART: Integer;
begin
  if H0 >= 0 then majAtm(H0);
  if Z0 < 1e-6 then
    Result := 0
  else
  begin
    // Integrate the refraction integral in the troposhere and stratosphere
    for K := 1 to 2 do
    begin
      REFP := -999.9;
      IT := 16;
      ISTART := 0;
      FE := 0;
      FO := 0;
      if K = 1 then
      begin
        // At the Observer
        FB := ATMO(1, Atm[5], N, DNDR);
        SK0 := N*Atm[5]*Sin(Z0*DGR);
        // At the Tropopause, troposphere side
        FF := ATMO(1, Atm[11], N, DNDR);
        Atm[0] := N;
        ZT := Arcsin(SK0/Atm[11]/N)/DGR;
        H := (ZT - Z0)/IT;
      end
      else
      begin
        // At the Tropopause, stratosphere side
        FB := ATMO(2, Atm[11], N, DNDR);
        // At the stratosphere limit
        FF := ATMO(2, RS, N, DNDR);
        H := (Arcsin(SK0/RS/N)/DGR - ZT)/IT;
      end;
      repeat
        REFO := REFP;
        for I := 1 to IT + ISTART - 1 do
        begin
          if (I = 1) and (K = 1) then
          begin
            Z := Z0 + H;
            R := Atm[5];
          end
          else
            if I = 1 then
            begin
              Z := ZT + H;
              R := Atm[11];
            end
            else
              Z := Z + (1 + ISTART)*H;

          // Given the Zenith distance (Z) find R
          for J := 1 to 4 do
          begin
            ATMO(K, R, N, DNDR);
            R := R - (R*N - SK0/Sin(Z*DGR))/(N + R*DNDR);
          end;

          // Find Refractive index and Integrand at R
          if (ISTART = 0) and (I mod 2 = 0) then
            FE := FE + ATMO(K, R, N, DNDR)
          else
            FO := FO + ATMO(K, R, N, DNDR);
        end;

        // Evaluate the integrand using Simpson’s Rule
        REFP := H*(FB + 4*FO + 2*FE + FF)/3;

        // Préparation pour une éventuelle nouvelle boucle
        IT := (1 + ISTART)*IT;
        ISTART := 1;
        H := H/2;
        FE := FE + FO;
        FO := 0;
      until abs(REFP - REFO) < 0.5*EPS/3600;

      if K = 1 then
        REFT := REFP;
    end;
    Result := REFT + REFP;
  end;
end;

function haVhv(ha, H0: Real): Real; // Réfraction: passe de la hauteur apparente à la hauteur vraie
begin
  // Result := ho - 1/60/Tan(DegToRad(ho + 7.31/(ho + 4.41)));
  Result := ha - IntRef(90 - ha, H0);
end;

function hvVha(hv, H0: Real): Real;  // Réfraction: passe de la hauteur vraie à la hauteur apparente
const
  epsi = 1e-6;
  ni = 10;
var
  n, ie: Integer;
  he, hs: Array[0..1] of Real;
begin
  if hv > 90 - 1e-6 then
    Result := hv
  else
  begin
    he[0] := hv;
    hs[0] := haVhv(he[0], H0) - hv;   // Initialise Atm
    he[1] := he[0] - hs[0];
    hs[1] := haVhv(he[1], -1) - hv;   // Il n'est plus nécessaire d'initialiser Atm
    // Boucle d'itération
    for n := 0 to ni do
    begin
      Result := (he[0]*hs[1] - he[1]*hs[0])/(hs[1] - hs[0]);
  //    ShowMessage(IntToStr(n) + #13#10
  //              + FloatToStr(he[0]) + ' / ' + FloatToStr(Result - he[0]) + #13#10
  //              + FloatToStr(he[1]) + ' / ' + FloatToStr(Result - he[1]) + #13#10
  //              + FloatToStr(Result) + ' / ' + FloatToStr(haVhv(Result)));
      if SameValue(Result, he[0], epsi) or SameValue(Result, he[1], epsi) then break;
      ie := 0;
      if Abs(Result - he[1]) > Abs(Result - he[0]) then ie := 1;
      he[ie] := Result; hs[ie] := haVhv(he[ie], -1) - hv;
    end;
    if n = ni then MessDlgPos(MHaut[0] + ' (hvVha)', mtError, Form3.Left + xms, Form3.Top + xms);
  end;
end;

function alVde(al: Real): Real;
var
  N: Real = 0;
  NS: Real = 0;
  DNDR: Real = 0;
begin
  majAtm(al);
  ATMO(1, RTT + al, N, DNDR);
  ATMO(1, RTT, NS, DNDR);
  Result := RadToDeg(Arccos(RTT*NS/(RTT + al)/N));
end;

function deVal(de: Real): Real;  // Dépression: passe de la dépression à l'altitude d'observation
const
  // kde = 0.02943;
  kde = 0.0295;           // Valeur des éphémérides nautiques
  epsi = 1e-3;
  ni = 5;
var
  n, ie: Integer;
  ale, als: Array[0..1] of Real;
begin
  ale[0] := Sqr(de/kde);
  als[0] := alVde(ale[0]) - de;
  ale[1] := 1.1*ale[0];
  als[1] := alVde(ale[1]) - de;
  // Boucle d'itération
  for n := 0 to ni do
  begin
    Result := (ale[0]*als[1] - ale[1]*als[0])/(als[1] - als[0]);
//    ShowMessage(IntToStr(n) + #13#10
//              + FloatToStr(ale[0]) + ' / ' + FloatToStr(Result - ale[0]) + #13#10
//              + FloatToStr(ale[1]) + ' / ' + FloatToStr(Result - ale[1]) + #13#10
//              + FloatToStr(Result) + ' / ' + FloatToStr(alVde(Result)));
    if SameValue(Result, ale[0], epsi) or SameValue(Result, ale[1], epsi) then break;
    ie := 0;
    if Abs(Result - ale[1]) > Abs(Result - ale[0]) then ie := 1;
    ale[ie] := Result;
    als[ie] := alVde(ale[ie]) - de;
  end;
  if n = ni then MessDlgPos(MHaut[0] + ' (deVal)', mtError, Form3.Left + xms, Form3.Top + xms);
end;

procedure TForm3.ChkHiClick(Sender: TObject);
begin
  ChkClick(12);
end;

procedure TForm3.ChkCoClick(Sender: TObject);
begin
  ChkClick(13);
end;

procedure TForm3.ChkHeClick(Sender: TObject);
begin
  ChkClick(14);
end;

procedure TForm3.EdHiExit(Sender: TObject);
begin
  EdExit(12);
end;

procedure TForm3.EdCoExit(Sender: TObject);
begin
  EdExit(13);
end;

procedure TForm3.EdHeExit(Sender: TObject);
begin
  EdExit(14);
end;

procedure TForm3.UniHiChange(Sender: TObject);
begin
  UniChange(12);
end;

procedure TForm3.UniCoChange(Sender: TObject);
begin
  UniChange(13);
end;

procedure TForm3.ChkHoThClick(Sender: TObject);
begin
  if ChkHoTh.Checked then
  begin
    LiC[14].Visible := False;
    Label6.Visible := True;
    LiC[14].Checked := True;
    if LiE[14].Text = '' then
    begin
      LiE[14].Text := '0'; LiE[14].Modified := True;
      EdExit(14);
    end;
    LiE[14].Color := Tcolor(Tri1);
  end
  else
  begin
    LiC[14].Visible := True;
    Label6.Visible := False;
    LiE[14].Color := clTriA;
  end;
end;

procedure TForm3.FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState
  );
// Outil de mise au point pour afficher les valeurs calculées en pleine définition
begin
  if (Key = Ord('Y')) and (ssCtrl in Shift) then
  begin
    quits();        // Pour quitter un éventuel champ de saisie
    MessDlgPos(MTri[15]
        + #13#10'12 : ' + BoolToStr(Tbol[12], True)+ ', ' + FloatToStr(Tvar[12, 0])
        + #13#10'13 : ' + BoolToStr(Tbol[13], True)+ ', ' + FloatToStr(Tvar[13, 0])
        + #13#10'14 : ' + BoolToStr(Tbol[14], True)+ ', ' + FloatToStr(Tvar[14, 0]),
        mtInformation, Left + Width, Top);
  end;
end;

procedure TForm3.BtCalculClick(Sender: TObject);
const
  subh = -3.1;           // Limite inférieure de l'angle réel de l'astre
var
  i, p2, cf: Integer;
  bds, dep, col: Real;
  hi, hv: Real;
begin
  quits();                // Pour quitter un éventuel champ de saisie
  case CmbSo.ItemIndex of // Bord ou centre du Soleil
    0: bds := 0;          // Centre
    1: if Tbol[6] then    // Bord inférieur
        bds := ephe(Tvar[6, 0], 180, sRs)
      else
        bds := 16/60;
    2: if Tbol[6] then    // Bord supérieur
        bds := -ephe(Tvar[6, 0], 180, sRs)
      else
        bds := -16/60;
  end;
  if ChkHoTh.Checked and (LiE[14].Text = '') then
  begin
    LiE[14].Text := '0'; LiE[14].Modified := True;
    EdExit(14);
  end;
  p2 := 1;
  cf := 0;
  for i := 12 to 14 do    // Calcul de la configuration
  begin
    if Tbol[i] then cf := cf + p2;
    p2 := 2*p2;
  end;
  if Tbol2[hr] then cf := cf + p2;
  case cf of
    0, 1, 2, 3, 4, 5, 6, 8, 9, 10, 12:
      MessDlgPos(MTri[18], mtError, Left + xms, Top + yms)
    else begin
      for i := 12 to 14 do                // Décoche la case s'il n'y a pas de valeur saisie
        if LiC[i].Checked and not Tbol[i] then
          LiC[i].Checked := False;
      if LiC2[hr].Checked and not Tbol2[hr] then
        LiC2[hr].Checked := False;
      case cf of    // Traitement de la configuration
        14, 13: begin
          hv := modr(hr) - bds;  // Calcul de la hauteur réelle de la partie du Soleil concerné
          if ChkPrlx.Checked then
            hv := hv - 2.44e-3*Cos(DegToRad(hv));
          dep := alVde(Tvar[14, 0]);
          if hv > subh then        // Correction de cette hauteur si résultat potentiellement correct
            hi := hvVha(hv, Tvar[14, 0])
          else
            hi := hv;
          if hi < -dep  then
            MessDlgPos(MHaut[1], mtError, Left + xms, Top + yms) // Observation impossible
          else
          begin
            if not ChkHoTh.Checked then
              hi := hi + dep;
            if ChkDbl.Checked then
              hi := 2*hi;
            case cf of
              14: begin   // Calcul de la hauteur observée ou instrumentale hi
                Tvar[12,0] := hi  - Tvar[13, 0];
                afrt(12);
              end;
              13: begin   // Calcul de la collimation co
                col := hi - Tvar[12, 0];
                if (col < -1) or (col > 1) then
                  MessDlgPos(MHaut[3], mtError, Left + xms, Top + yms)
                else
                begin
                  Tvar[13, 0] := col;
                  afrt(13);
                end;
              end;
            end;
          end;
        end;
        11, 7: begin
          hi := Tvar[12, 0] + Tvar[13, 0];
          if ChkDbl.Checked then
            hi := hi/2;
          dep := alVde(Tvar[14, 0]);
          if not ChkHoTh.Checked and (hi < 0) then
            MessDlgPos(MHaut[2], mtError, Left + xms, Top + yms)
          else
            if hi < -dep then
              MessDlgPos(MHaut[2], mtError, Left + xms, Top + yms)
            else
              case cf of
                7: begin    // Calcul de la hauteur géocentrique hv
                  if not ChkHoTh.Checked then
                    hi := hi - dep;
                  hv := haVhv(hi, Tvar[14, 0]) + bds;
                  if ChkPrlx.Checked then
                    hv := hv + 2.44e-3*Cos(DegToRad(hv));
                  if hv > 90 then
                    MessDlgPos(MHaut[4], mtError, Left + xms, Top + yms)
                  else
                  begin
                    Tvar[hr,0] := hv;
                    afrt(hr);
                  end;
                end;
                11: begin   // Calcul de la hauteur de l'oeil ho
                  if ChkHoTh.Checked then
                    MessDlgPos(MTri[19], mtError, Left + xms, Top + yms)
                  else
                  begin
                    hv := modr(hr) - bds;  // Calcul de la hauteur réelle de la partie du Soleil concerné
                    if ChkPrlx.Checked then
                      hv := hv - 2.44e-3*Cos(DegToRad(hv));
                    if hv < subh then
                      MessDlgPos(MHaut[3], mtError, Left + xms, Top + yms)
                    else
                    begin
                      dep := hi - hvVha(hv, 0);  // Première estimation
                      if (dep < 0) or (dep > alVde(1.1*altm)) then
                        MessDlgPos(MHaut[3], mtError, Left + xms, Top + yms)
                      else
                      begin
                        dep := hi - hvVha(hv, deVal(dep)); // Calcul final
                        if (dep < 0) or (dep > alVde(altm)) then
                          MessDlgPos(MHaut[3], mtError, Left + xms, Top + yms)
                        else
                        begin
                          Tvar[14,0] := deVal(dep);
                          afrt(14);
                        end;
                      end;
                    end;
                  end;
                end;
              end;
        end;
        else
          MessDlgPos(MTri[19], mtError, Left + xms, Top + yms);
      end;
    end;
  end;
end;

procedure TForm3.BtRazClick(Sender: TObject);
var
  i: Integer;
begin
  quits();                // Pour quitter un éventuel champ de saisie
  for i := 12 to 14 do
    if not Tbol[i] then
    begin
      LiE[i].Text := '';
      LiE[i].Modified := False;
    end;
  if not Tbol2[hr] then
  begin
    LiE[hr].Text := '';
    LiE[hr].Modified := False;
    Tbol[hr] := False;
  end;
end;

function TForm3.{%H-}FormHelp(Command: Word; Data: PtrInt; var CallHelp: Boolean
  ): Boolean;
begin
  Form1.MaideClick(nil);
end;

procedure TForm3.BtParClick(Sender: TObject);
var
  Fpar: TForm;
  EdTemp, EdPres, EdLamb, EdHumi: TEdit;
  Dim: Tpoint;
  i, esph, espv, finh, finv: Integer;
  vl: Real;
begin
  Fpar := TForm.Create(Application);
  with Fpar do
    try
      BorderStyle := bsDialog;
      Caption := MHaut[5];
      Left := Form3.Left + xms;
      Top := Form3.Top + yms;
      Dim := DimT('X', Canvas);
      esph := MulDiv(4, Dim.X, 2);
      espv := MulDiv(1, Dim.Y, 2);
      with TLabel.Create(Fpar) do
      begin
        Parent := Fpar;
        Caption := MHaut[6];
        Left := esph;
        Top := MulDiv(3, espv, 2);
        finh := Left + DimT(Caption, Canvas).X;
      end;
      EdTemp := TEdit.Create(Fpar);
      with EdTemp do
      begin
        Parent := Fpar;
        Top := espv;
        Text := FloatToStr(temp);
        finv := Top + Height;
      end;
      with TLabel.Create(Fpar) do
      begin
        Parent := Fpar;
        Caption := MHaut[7];
        Left := esph;
        Top := finv + MulDiv(3, espv, 2);
        finh := Max(finh, Left + DimT(Caption, Canvas).X);
      end;
      EdPres := TEdit.Create(Fpar);
      with EdPres do
      begin
        Parent := Fpar;
        Top := finv + espv;
        Text := FloatToStr(pres);
        finv := Top + Height;
      end;
      with TLabel.Create(Fpar) do
      begin
        Parent := Fpar;
        Caption := MHaut[8];
        Left := esph;
        Top := finv + MulDiv(3, espv, 2);
        finh := Max(finh, Left + DimT(Caption, Canvas).X);
      end;
      EdLamb := TEdit.Create(Fpar);
      with EdLamb do
      begin
        Parent := Fpar;
        Top := finv + espv;
        Text := FloatToStr(lamb);
        finv := Top + Height;
      end;
      with TLabel.Create(Fpar) do
      begin
        Parent := Fpar;
        Caption := MHaut[9];
        Left := esph;
        Top := finv + MulDiv(3, espv, 2);
        finh := Max(finh, Left + DimT(Caption, Canvas).X);
      end;
      EdHumi := TEdit.Create(Fpar);
      with EdHumi do
      begin
        Parent := Fpar;
        Top := finv + espv;
        Text := FloatToStr(humi);
        finv := Top + Height;
      end;
      i := finh + esph;
      EdTemp.Left := i;
      finh := i + EdTemp.Width;
      EdPres.Left := i;
      finh :=  Max(finh, i + EdPres.Width);
      EdLamb.Left := i;
      finh :=  Max(finh, i + EdLamb.Width);
      EdHumi.Left := i;
      finh :=  Max(finh, i + EdHumi.Width);
      with TButton.Create(Fpar) do
      begin
        Parent := Fpar;
        Caption := MTri[12];
        Default := True;
        Left := (finh + esph - Width) div 2 ;
        Top := finv + espv;
        finv := Top + Height;
        ModalResult := mrOK;
      end;
      ClientWidth := finh + esph;
      ClientHeight := finv + espv;
      if ShowModal = mrOK then
      begin
        Val(EdTemp.Text, vl, i);
        if (i = 0) and (vl >= -20) and (vl <= 40) then
          temp := vl
        else
          MessDlgPos(MHaut[10], mtError, Left + xms, Top + yms);
        Val(EdPres.Text, vl, i);
        if (i = 0) and (vl >= 500) and (vl <= 1050) then
          pres := vl
        else
          MessDlgPos(MHaut[11], mtError, Left + xms, Top + yms);
        Val(EdLamb.Text, vl, i);
        if (i = 0) and (vl >= 400) and (vl <= 800) then
          lamb := vl
        else
          MessDlgPos(MHaut[12], mtError, Left + xms, Top + yms);
        Val(EdHumi.Text, vl, i);
        if (i = 0) and (vl >= 0) and (vl <= 100) then
          humi := vl
        else
          MessDlgPos(MHaut[13], mtError, Left + xms, Top + yms);
      end;
    finally
      Fpar.Free;
    end;
end;

procedure TForm3.FormCreate(Sender: TObject);
var
  i: Integer;
begin
  if Elng <> '' then clng(3); // Met en place le changement de langue pour Form3
  LiC[12] := ChkHi; LiC[13] := ChkCo; LiC[14] := ChkHe;
  LiE[12] := EdHi; LiE[13] := EdCo; LiE[14] := EdHe;
  LiB[12] := UniHi; LiB[13] := UniCo;
  for i := 12 to 14 do
    Tbol[i] := False;
  for i := 12 to 13 do
  begin
    Unt[i] := nunt(LiB[i].Items[0]);
    LiE[i].Hint := chnt(i);
  end;
  Unt[14] := 0;  // En fait en mètre, mais pour obtenir directement la valeur dans Tvar[14, 0]
  {$IF DEFINED(LINUX) OR DEFINED(DARWIN)}
  // Ajustement de la taille des Edit pour afficher entièrement les valeurs
  Width := Width + aged;
  Label3.Left := Label3.Left + aged div 2;
  Label4.Left := Label4.Left + aged;
  Label5.Left := Label5.Left + aged;
  CmbSo.Width := CmbSo.Width + aged;
  BtCalcul.Width := BtCalcul.Width + aged;
  for i := 12 to 14 do
  begin
    LiE[i].Width := LiE[i].Width + aged;
    if i < 14 then LiB[i].Left := LiB[i].Left + aged;
  end;
  {$ENDIF}
  {$IFDEF DARWIN}
  // Ajustement de la taille des ComboBox pour afficher entièrement les unités
  Width := Width + agcb;
  Label4.Left := Label4.Left + agcb div 2;
  BtCalcul.Width := BtCalcul.Width + agcb;
  for i := 12 to 13 do
    LiB[i].Width := LiB[i].Width + agcb;
  {$ENDIF}
end;

end.

